#!/usr/bin/env python

#****************************************************************************
#** $Id: tablestatistics.py,v 1.1 2002/06/19 07:56:07 phil Exp $
#**
#** Copyright (C) 1992-1998 Troll Tech AS.  All rights reserved.
#**
#** This file is part of an example program for PyTQt.  This example
#** program may be used, distributed and modified without limitation.
#**
#*****************************************************************************/

import sys
import os
from qt import *
from qttable import *

TRUE  = 1
FALSE = 0

# column constants
TB_FILE = 0
TB_SIZE = 1
TB_FLAG = 2
TB_COLS = 3

dirs = (
    "kernel",
    "tools",
    "widgets",
    "dialogs",
    "xml",
    "table",
    "network",
    "opengl",
    "canvas",
)

class Table(TQTable):
    def __init__(self):
        TQTable.__init__(self, 0, TB_COLS)
        self.setSorting(TRUE)
        self.horizontalHeader().setLabel(TB_FILE, self.tr("File"))
        self.horizontalHeader().setLabel(TB_SIZE, self.tr("Size (bytes)"))
        self.horizontalHeader().setLabel(TB_FLAG, self.tr("Use in Sum"))
        self.initTable()
        self.adjustColumn(TB_FILE)

        # if the user edited something we might need to recalculate the sum
        self.connect(self, SIGNAL("valueChanged(int, int)"), self.recalcSum)

    def initTable(self):
        # read all the TQt source and header files into a list
        all = []
        qtdir = os.getenv("QTDIR")
        for i in dirs:
            dir = TQDir(os.path.join(qtdir, "src", i))
            lst = TQStringList(dir.entryList("*.cpp; *.h"))
            for f in lst:
                if f.contains("moc"):
                    continue
                all.append(os.path.join(i, str(f)))

        # set the number of rows we'll need for the table
        self.setNumRows(len(all) + 1)

        i = 0
        sum = 0L
        # insert the data into the table
        for it in all:
            self.setText(i, TB_FILE, it)
            f = TQFile(os.path.join(qtdir, "src", str(it)))
            self.setText(i, TB_SIZE, str(f.size()))
            ci = ComboItem(self, TQTableItem.WhenCurrent)
            self.setItem(i, TB_FLAG, ci)
            i = i + 1
            sum += f.size()
        self.displaySum(sum)

    def recalcSum(self, dummy, col):
        # only recalc if a value in the second or third column changed
        if col < TB_SIZE or col > TB_FLAG:
            return

        sum = 0L
        for i in range(self.numRows()-1):
            if str(self.text(i, TB_FLAG)) == "No":
                continue
            sum += long(str(self.text(i, TB_SIZE)))
        self.displaySum(sum)

    def displaySum(self, sum):
        # insert calculated data
        i1 = TableItem(self, TQTableItem.Never, self.tr("Sum"))
        self.setItem(self.numRows()-1, TB_FILE, i1)
        i2 = TableItem(self, TQTableItem.Never, str(sum))
        self.setItem(self.numRows()-1, TB_SIZE, i2)

    def sortColumn(self, col, ascending, wholeRows):
        # sum row should not be sorted, so get rid of it for now
        self.clearCell(self.numRows()-1, TB_FILE)
        self.clearCell(self.numRows()-1, TB_SIZE)
        # do sort
        TQTable.sortColumn(self, col, ascending, TRUE)
        # re-insert sum row
        self.recalcSum(0, TB_SIZE)


class TableItem(TQTableItem):
    def __init__(self, *args):
        apply(TQTableItem.__init__, (self,) + args)

    def paint(self, p, cg, cr, selected):
        g = TQColorGroup(cg)
        # last row is the sum row - we want to make it more visible by
        # using a red background
        if self.row() == self.table().numRows()-1:
            g.setColor(TQColorGroup.Base, TQColor("red"))
            TQTableItem.paint(self, p, g, cr, selected)


class ComboItem(TQTableItem):
    def __init__(self, t, et):
        TQTableItem.__init__(self, t, et, "Yes")
        self.cb = None
        # we do not want this item to be replaced
        self.setReplaceable(FALSE)

    def createEditor(self):
        # create an editor - a combobox in our case
        self.cb = TQComboBox(self.table().viewport())
        TQObject.connect(self.cb, SIGNAL("activated(int)"), 
                        self.table(), SLOT("doValueChanged()"))
        self.cb.insertItem("Yes")
        self.cb.insertItem("No")
        # and initialize it
        if str(self.text()) == "No":
            self.cb.setCurrentItem(1)
        else:
            self.cb.setCurrentItem(0)
        return self.cb

    def setContentFromEditor(self, w):
        # the user changed the value of the combobox, so synchronize the
        # value of the item (its text), with the value of the combobox
        if w.inherits("TQComboBox"):
            self.setText(w.currentText())
        else:
            TQTableItem.setContentFromEditor(self, w)

    def setText(self, s):
        # initialize the combobox from the text
        if self.cb:
            if str(s) == "No":
                self.cb.setCurrentItem(1)
            else:
                self.cb.setCurrentItem(0)
        TQTableItem.setText(self, s)


if __name__ == '__main__':
    app = TQApplication(sys.argv)

    t = Table()
    t.setCaption("Statistics")
    t.show()
    app.setMainWidget(t)
    app.exec_loop()
